﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Xml.Linq;

namespace ClassLibrary5
{
    public class AbbyySvgRegionLearner
    {
        private readonly ICollection<XDocument> _svgDocuments;

        public readonly string _documentName;

        public AbbyySvgRegionLearner(ICollection<XDocument> svgDocuments, string documentName)
        {
            _svgDocuments = svgDocuments;
            _documentName = documentName;
        }

        private IEnumerable<XElement> GetPages(XDocument sourceData)
        {
            var svgNamespace = "{http://www.w3.org/2000/svg}";
            //svgNamespace = "";
            var contentExtractorNode = sourceData.Element("ocr-extract");

            if (contentExtractorNode != null)
            {
                var pagesElement = contentExtractorNode.Element("pages");

                if (pagesElement != null)
                {
                    var pages = pagesElement.Elements("page");

                    return pages.Select(page => page.Element(svgNamespace +"svg")).ToArray();
                }
            }

            return new XElement[0];
        }

        public void AddFieldLocation(string fieldName, int pageNumber, ICollection<Element> textElements)
        {

        }

        private IEnumerable<Element> GetElements()
        {
            foreach (var svgDocument in _svgDocuments)
            {
                var guid = Guid.NewGuid();
                var pageNumber = 1;

                foreach (var page in GetPages(svgDocument))
                {
                    foreach (var element in page.Elements())
                    {
                        yield return new Element()
                        {
                            DocumentGuid = guid,
                            DocumentType = _documentName,
                            PageNumber = pageNumber,
                            Text = element.Value,
                            X = Convert.ToInt32(element.Attribute("x").Value),
                            Y = Convert.ToInt32(element.Attribute("y").Value),
                            Suspicious = Convert.ToBoolean(element.Attribute("suspicious").Value),
                            Scale = Convert.ToInt32(page.Attribute("width").Value),
                        };
                    }

                    pageNumber++;
                }
            }
        }

        public IEnumerable<IGrouping<Guid, IGrouping<object, Element>>> GetDocuments()
        {
            return GetElements().GroupBy(x => new { x.DocumentGuid, x.PageNumber }).GroupBy(x => x.Key.DocumentGuid).ToList();
        }
    }

    public class Element : IEquatable<Element>
    {
        public Guid DocumentGuid { get; set; }
        public string DocumentType { get; set; }
        public string Text { get; set; }
        public int X { get; set; }
        public int Y { get; set; }
        public bool Suspicious { get; set; }
        public int Scale { get; set; }
        public int PageNumber { get; set; }
        public override string ToString()
        {
            return $"{Text} X:{X} Y:{Y}";
        }

        public bool Equals(Element other)
        {
            if (DocumentGuid != other.DocumentGuid) return false;
            if (DocumentType != other.DocumentType) return false;
            if (Text != other.Text) return false;
            if (X != other.X) return false;
            if (Y != other.Y) return false;
            if (Suspicious != other.Suspicious) return false;
            if (PageNumber != other.PageNumber) return false;

            return true;
        }

        public override int GetHashCode()
        {
            return
                DocumentGuid.GetHashCode() * 17 +
                (DocumentType != null ? DocumentType.GetHashCode() * 17 : 0) +
                Text.GetHashCode() * 17 +
                X.GetHashCode() * 17 +
                Y.GetHashCode() * 17 +
                Suspicious.GetHashCode() * 17 +
                PageNumber.GetHashCode() * 17
            ;
        }
    }
}