﻿using ClassLibrary5;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Windows.Forms;
using System.Xml;
using System.Xml.Linq;
using System.Xml.Serialization;

namespace WindowsFormsApplication2
{
    public partial class Form1 : Form
    {
        private IEnumerable<IGrouping<Guid, IGrouping<object, Element>>> _documents { get; set; }

        private List<string> _elementTypes { get; set; }

        public List<Datapoint> _datapoints { get; set; } = new List<Datapoint>();

        private List<string> _dictionaryWords { get; set; } = File.ReadAllLines("words_alpha.txt").ToList();

        public Form1()
        {
            InitializeComponent();
            InitializeElementTypes();
            lstDataPoints.DataSource = _datapoints;

            var doc = XDocument.Load(@"C:\Users\willh\Documents\Visual Studio 2015\Projects\MaximiseIT.Clara\UnitTests\TestData\SampleDocuments\MortgageLiveLink\Mortgage Application.svg");

            var abbyySvgRegionLearner = new AbbyySvgRegionLearner(new List<XDocument>() { doc }, "Mortgage Application");

            _documents = abbyySvgRegionLearner.GetDocuments().ToList();

            cmbDocuments.DataSource = _documents;
            cmbDocuments.DisplayMember = "Key";

            numLocalityDistance.Maximum = 10000;
            numLocalityDistance.Value = 125;

            tabMain.SelectedIndexChanged += tabMain_SelectedIndexChanged;
        }

        private void InitializeElementTypes()
        {
            _elementTypes = new List<string>() {

                "First Name",
                "Last Name",
                "First Name 2",
                "Last Name 2",
                "Postcode",
                "Postcode 2",
                "Account Number",
                "Customer Number",
                "Date of Birth",
                "Date of Birth 2",
            };

            cmbElementTypes.DataSource = _elementTypes;
        }

        private void cmbDocuments_SelectedIndexChanged(object sender, EventArgs e)
        {
            cmbPages.DataSource = (cmbDocuments.SelectedItem as IEnumerable<IGrouping<object, Element>>).ToList();
            cmbPages.DisplayMember = "Key";
        }

        private void cmbPages_SelectedIndexChanged(object sender, EventArgs e)
        {
            cmbElements.DataSource = (cmbPages.SelectedItem as IEnumerable<Element>).ToList();
        }

        private void txtSearch_TextChanged(object sender, EventArgs e)
        {
            cmbElements.DataSource = (cmbPages.SelectedItem as IEnumerable<Element>)
                .Where(x => x.Text.ToLower().Contains(txtSearch.Text)).ToList();
        }

        private void cmbElements_SelectedIndexChanged(object sender, EventArgs e)
        {
            lstLocality.DataSource = (cmbPages.SelectedItem as IEnumerable<Element>)
                .Where(IsElementInLocality)
                .Where(IsElementDictionaryWord)
                .Where(x => !IsElementSuspicious(x))
                .ToList();
        }

        private bool IsElementInLocality(Element element)
        {
            var selectedElement = (cmbElements.SelectedItem as Element);

            var localityDistance = numLocalityDistance.Value;

            return
                selectedElement.X + localityDistance > element.X &&
                selectedElement.X - localityDistance < element.X &&
                selectedElement.Y + localityDistance > element.Y &&
                selectedElement.Y - localityDistance < element.Y &&
                selectedElement != element;
        }
        
        private bool IsElementDictionaryWord(Element element)
        {
            return _dictionaryWords.Any(x => element.Text.Contains(x));
        }

        private bool IsElementSuspicious(Element element)
        {
            return element.Suspicious;
        }

        private void numLocalityDistance_ValueChanged(object sender, EventArgs e)
        {
            cmbElements_SelectedIndexChanged(sender, e);
        }

        private void btnSave_Click(object sender, EventArgs e)
        {
            var datapoint = new Datapoint()
            {
                ElementType = (cmbElementTypes.SelectedItem as string),
                ElementLocalities = (lstLocality.DataSource as IEnumerable<Element>).Select(x => new Element()
                {
                    DocumentGuid = x.DocumentGuid,
                    DocumentType = x.DocumentType,
                    PageNumber = x.PageNumber,
                    Suspicious = x.Suspicious,
                    Text = x.Text,
                    X = (cmbElements.SelectedItem as Element).X - x.X,
                    Y = (cmbElements.SelectedItem as Element).Y - x.Y,
                })
                .ToList()
            };

            _datapoints.Add(datapoint);

            if (!Directory.Exists("Mortgage Application"))
            {
                Directory.CreateDirectory("Mortgage Application");
            }

            SerializeObject(datapoint, $"Mortgage Application/{ DateTime.Now.ToFileTimeUtc().ToString()}.xml");
            lstDataPoints.DataSource = _datapoints.ToList();
            lstDataPoints.Refresh();
        }
        
        private void lstDataPoints_SelectedIndexChanged(object sender, EventArgs e)
        {
            lstDataPointLocalities.DataSource = (lstDataPoints.SelectedItem as Datapoint).ElementLocalities.ToList();
        }

        /// <summary>
        /// Serializes an object.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="serializableObject"></param>
        /// <param name="fileName"></param>
        public void SerializeObject<T>(T serializableObject, string fileName)
        {
            if (serializableObject == null) { return; }

            try
            {
                XmlDocument xmlDocument = new XmlDocument();
                XmlSerializer serializer = new XmlSerializer(serializableObject.GetType());
                using (MemoryStream stream = new MemoryStream())
                {
                    serializer.Serialize(stream, serializableObject);
                    stream.Position = 0;
                    xmlDocument.Load(stream);
                    xmlDocument.Save(fileName);
                    stream.Close();
                }
            }
            catch (Exception ex)
            {
                //Log exception here
            }
        }


        /// <summary>
        /// Deserializes an xml file into an object list
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="fileName"></param>
        /// <returns></returns>
        public T DeSerializeObject<T>(string fileName)
        {
            if (string.IsNullOrEmpty(fileName)) { return default(T); }

            T objectOut = default(T);

            try
            {
                XmlDocument xmlDocument = new XmlDocument();
                xmlDocument.Load(fileName);
                string xmlString = xmlDocument.OuterXml;

                using (StringReader read = new StringReader(xmlString))
                {
                    Type outType = typeof(T);

                    XmlSerializer serializer = new XmlSerializer(outType);
                    using (XmlReader reader = new XmlTextReader(read))
                    {
                        objectOut = (T)serializer.Deserialize(reader);
                        reader.Close();
                    }

                    read.Close();
                }
            }
            catch (Exception ex)
            {
                //Log exception here
            }

            return objectOut;
        }


        #region Read

        private void tabMain_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (tabMain.SelectedTab.Name == "tabRead")
            {
                cmbLearnedDocuments.DataSource = Directory.GetDirectories(Directory.GetCurrentDirectory()).Select(x => x.Split(new[] { "\\" }, StringSplitOptions.None).Last()).ToList();
            }
        }

        private void cmbLearnedDocuments_SelectedIndexChanged(object sender, EventArgs e)
        {
            var _readDataPoints = new List<Datapoint>();

            foreach (var filePath in Directory.GetFiles(cmbLearnedDocuments.SelectedItem.ToString()))
            {
                _readDataPoints.Add(DeSerializeObject<Datapoint>(filePath));
            }

            lstAvailableElementTypes.DataSource = _readDataPoints.GroupBy(x => x.ElementType).ToList();
            lstAvailableElementTypes.DisplayMember = "Key";
        }

        private void lstAvailableElementTypes_SelectedIndexChanged(object sender, EventArgs e)
        {
            var datapoints = (lstAvailableElementTypes.SelectedItem as IGrouping<string, Datapoint>);

            var probableLocations = new List<Element>();

            foreach (var element in datapoints.SelectMany(x => x.ElementLocalities))
            {
                probableLocations.AddRange(
                    _documents.First().ToList()[element.PageNumber - 1]
                    .Where(x => x.Text == element.Text)
                    .Select(x => new Element() {
                        X = x.X + element.X,
                        Y = x.Y + element.Y,
                        PageNumber = x.PageNumber,
                        Scale = x.Scale
                    })
                );
            }

            foreach (var probableLocation in probableLocations)
            {
                probableLocation.Text = _documents.First().ToList()[probableLocation.PageNumber - 1]
                    .Where(x => IsElementInLocality2(x, probableLocation)).FirstOrDefault()?.Text;
            }

            probableLocations.RemoveAll(x => x.Text == null);
            
            lstProbableLocations.DataSource = probableLocations.GroupBy(x => x.Text).OrderByDescending(x => x.Count()).ToList();
            lstProbableLocations.DisplayMember = "Key";
        }

        private bool IsElementInLocality2(Element element, Element element2)
        {
            var localityDistance = 1;

            return
                (element2.X + localityDistance / element2.Scale * element.Scale) >= element.X &&
                (element2.X - localityDistance / element2.Scale * element.Scale) <= element.X &&
                (element2.Y + localityDistance / element2.Scale * element.Scale) >= element.Y &&
                (element2.Y - localityDistance / element2.Scale * element.Scale) <= element.Y;
        }

        #endregion

    }

    public class Datapoint
    {
        public string ElementType { get; set; }

        public List<Element> ElementLocalities { get; set; }

        public override string ToString()
        {
            return $"{ElementType} ({ElementLocalities.Count} localities)";
        }
    }
}
